
///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                               CPU Control                                 //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "..\include.h"

#define USE8BITTAB	// use 8-bit tables (faster, but requires more memory)
#define VERIFYTAB	// verify tables (or print tables otherwise)

///////////////////////////////////////////////////////////////////////////////
// 8-bit tables

#ifdef USE8BITTAB

#define BITTABORD 8					// number of bits of table size
#define BITTABNUM (1 << BITTABORD)	// bit table size

// reverse bit table
const u8 RevBitsTab[256] = {
 0x00, 0x80, 0x40, 0xC0, 0x20, 0xA0, 0x60, 0xE0, 0x10, 0x90, 0x50, 0xD0, 0x30, 0xB0, 0x70, 0xF0,
 0x08, 0x88, 0x48, 0xC8, 0x28, 0xA8, 0x68, 0xE8, 0x18, 0x98, 0x58, 0xD8, 0x38, 0xB8, 0x78, 0xF8,
 0x04, 0x84, 0x44, 0xC4, 0x24, 0xA4, 0x64, 0xE4, 0x14, 0x94, 0x54, 0xD4, 0x34, 0xB4, 0x74, 0xF4,
 0x0C, 0x8C, 0x4C, 0xCC, 0x2C, 0xAC, 0x6C, 0xEC, 0x1C, 0x9C, 0x5C, 0xDC, 0x3C, 0xBC, 0x7C, 0xFC,
 0x02, 0x82, 0x42, 0xC2, 0x22, 0xA2, 0x62, 0xE2, 0x12, 0x92, 0x52, 0xD2, 0x32, 0xB2, 0x72, 0xF2,
 0x0A, 0x8A, 0x4A, 0xCA, 0x2A, 0xAA, 0x6A, 0xEA, 0x1A, 0x9A, 0x5A, 0xDA, 0x3A, 0xBA, 0x7A, 0xFA,
 0x06, 0x86, 0x46, 0xC6, 0x26, 0xA6, 0x66, 0xE6, 0x16, 0x96, 0x56, 0xD6, 0x36, 0xB6, 0x76, 0xF6,
 0x0E, 0x8E, 0x4E, 0xCE, 0x2E, 0xAE, 0x6E, 0xEE, 0x1E, 0x9E, 0x5E, 0xDE, 0x3E, 0xBE, 0x7E, 0xFE,
 0x01, 0x81, 0x41, 0xC1, 0x21, 0xA1, 0x61, 0xE1, 0x11, 0x91, 0x51, 0xD1, 0x31, 0xB1, 0x71, 0xF1,
 0x09, 0x89, 0x49, 0xC9, 0x29, 0xA9, 0x69, 0xE9, 0x19, 0x99, 0x59, 0xD9, 0x39, 0xB9, 0x79, 0xF9,
 0x05, 0x85, 0x45, 0xC5, 0x25, 0xA5, 0x65, 0xE5, 0x15, 0x95, 0x55, 0xD5, 0x35, 0xB5, 0x75, 0xF5,
 0x0D, 0x8D, 0x4D, 0xCD, 0x2D, 0xAD, 0x6D, 0xED, 0x1D, 0x9D, 0x5D, 0xDD, 0x3D, 0xBD, 0x7D, 0xFD,
 0x03, 0x83, 0x43, 0xC3, 0x23, 0xA3, 0x63, 0xE3, 0x13, 0x93, 0x53, 0xD3, 0x33, 0xB3, 0x73, 0xF3,
 0x0B, 0x8B, 0x4B, 0xCB, 0x2B, 0xAB, 0x6B, 0xEB, 0x1B, 0x9B, 0x5B, 0xDB, 0x3B, 0xBB, 0x7B, 0xFB,
 0x07, 0x87, 0x47, 0xC7, 0x27, 0xA7, 0x67, 0xE7, 0x17, 0x97, 0x57, 0xD7, 0x37, 0xB7, 0x77, 0xF7,
 0x0F, 0x8F, 0x4F, 0xCF, 0x2F, 0xAF, 0x6F, 0xEF, 0x1F, 0x9F, 0x5F, 0xDF, 0x3F, 0xBF, 0x7F, 0xFF,
};

// order bit table
const u8 OrdBitsTab[256] = {
 0x00, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
};

// high order bit table (order of valid bits from top position)
const u8 HighOrdBitsTab[256] = {
 0x00, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x03, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x02, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x03, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x01, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x03, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x02, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x03, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
 0x04, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08, 0x05, 0x08, 0x07, 0x08, 0x06, 0x08, 0x07, 0x08,
};

///////////////////////////////////////////////////////////////////////////////
// 4-bit tables

#else // USE8BITTAB

#define BITTABORD 4					// number of bits of table size
#define BITTABNUM (1 << BITTABORD)	// bit table size

// reverse bit table
const u8 RevBitsTab[16] = {
//  0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F
 0x00, 0x08, 0x04, 0x0C, 0x02, 0x0A, 0x06, 0x0E, 0x01, 0x09, 0x05, 0x0D, 0x03, 0x0B, 0x07, 0x0F,
};

// order bit table
const u8 OrdBitsTab[16] = {
//  0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F
 0x00, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
};

// high order bit table (order of valid bits from top position)
const u8 HighOrdBitsTab[16] = {
//  0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F
 0x00, 0x04, 0x03, 0x04, 0x02, 0x04, 0x03, 0x04, 0x01, 0x04, 0x03, 0x04, 0x02, 0x04, 0x03, 0x04,
};

#endif // USE8BITTAB

///////////////////////////////////////////////////////////////////////////////
// create or verify bit tables (returns FALSE on table error)

bool InitBitTab()
{
	u8 tab[BITTABNUM];
	int i, j, k, m;

	// reverse bit table
	for (i = 0; i < BITTABNUM; i++)
	{
		k = 0;
		m = i;
		for (j = BITTABORD; j > 0; j--)
		{
			k <<= 1;
			if (m & 1) k |= 1;
			m >>= 1;
		}
		tab[i] = (u8)k;
	}

	// verify or output reverse bit table
#ifdef VERIFYTAB

	for (i = 0; i < BITTABNUM; i++)
		if (tab[i] != RevBitsTab[i]) return false;

#else // VERIFYTAB

	printf(	"// reverse bit table\n"
			"const u8 RevBitsTab[%d] = {\n", BITTABNUM);
	for (i = 0; i < BITTABNUM; i++)
	{
		printf(" 0x%02X,", tab[i]);
		if ((i & 0xf) == 0xf) printf("\n");
	}
	printf(	"};\n\n");

#endif // VERIFYTAB

	// order bit table
	for (i = 0; i < BITTABNUM; i++)
	{
		k = BITTABORD;
		m = i;
		for (j = BITTABORD; j > 0; j--)
		{
			if ((m & (1 << (BITTABORD-1))) != 0) break;
			m <<= 1;
			k--;
		}
		tab[i] = (u8)k;
	}

	// verify or output order bit table
#ifdef VERIFYTAB

	for (i = 0; i < BITTABNUM; i++)
		if (tab[i] != OrdBitsTab[i]) return false;

#else // VERIFYTAB

	printf(	"// order bit table\n"
			"const u8 OrdBitsTab[%d] = {\n", BITTABNUM);
	for (i = 0; i < BITTABNUM; i++)
	{
		printf(" 0x%02X,", tab[i]);
		if ((i & 0xf) == 0xf) printf("\n");
	}
	printf(	"};\n\n");

#endif // VERIFYTAB

	// high order bit table
	for (i = 0; i < BITTABNUM; i++)
	{
		k = BITTABORD;
		m = i;
		for (j = BITTABORD; j > 0; j--)
		{
			if ((m & 1) != 0) break;
			m >>= 1;
			k--;
		}
		tab[i] = (u8)k;
	}

	// verify or output high order bit table
#ifdef VERIFYTAB

	for (i = 0; i < BITTABNUM; i++)
		if (tab[i] != HighOrdBitsTab[i]) return false;

#else // VERIFYTAB

	printf(	"// high order bit table (order of valid bits from top position)\n"
			"const u8 HighOrdBitsTab[%d] = {\n", BITTABNUM);
	for (i = 0; i < BITTABNUM; i++)
	{
		printf(" 0x%02X,", tab[i]);
		if ((i & 0xf) == 0xf) printf("\n");
	}
	printf(	"};\n\n");

#endif // VERIFYTAB

	return true;
}

///////////////////////////////////////////////////////////////////////////////
// reverse bits

#ifdef USE8BITTAB

u8 Reverse8(u8 val)
{
	return RevBitsTab[val];
}

u16 Reverse16(u16 val)
{
	u16 res = RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val];
	return res;
}

u32 Reverse32(u32 val)
{
	u32 res = RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val];
	return res;
}

u64 Reverse64(u64 val)
{
	u64 res = RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val & 0xff];
	res <<= 8; val >>= 8; res |= RevBitsTab[val];
	return res;
}

#else // USE8BITTAB

u8 Reverse8(u8 val)
{
	return RevBitsTab[val >> 4] | (RevBitsTab[val & 0xf] << 4);
}

u16 Reverse16(u16 val)
{
	u16 res = RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val];
	return res;
}

u32 Reverse32(u32 val)
{
	u32 res = RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val];
	return res;
}

u64 Reverse64(u64 val)
{
	u64 res = RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val & 0xf];
	res <<= 4; val >>= 4; res |= RevBitsTab[val];
	return res;
}

#endif // USE8BITTAB

///////////////////////////////////////////////////////////////////////////////
// get numer of bits of value (logarithm, returns position of highest bit + 1: 0..64)

#ifdef USE8BITTAB

u8 Bits8(u8 val)
{
	return OrdBitsTab[val];
}

u8 Bits16(u16 val)
{
	int i = 0;
	if (val >= 0x100)
	{
		i = 8;
		val >>= 8;
	}
	return OrdBitsTab[val] + i;
}

u8 Bits32(u32 val)
{
	int i = 0;
	if (val >= 0x10000)
	{
		i = 16;
		val >>= 16;
	}

	if (val >= 0x100)
	{
		i += 8;
		val >>= 8;
	}
	return OrdBitsTab[val] + i;
}

u8 Bits64(u64 val)
{
	if (val > 0xffffffff) return ::Bits32((u32)(val >> 32)) + 32;
	return ::Bits32((u32)val);
}

#else // USE8BITTAB

u8 Bits8(u8 val)
{
	int i = 0;
	if (val >= 0x10)
	{
		i = 4;
		val >>= 4;
	}
	return OrdBitsTab[val] + i;
}

u8 Bits16(u16 val)
{
	int i = 0;
	if (val >= 0x100)
	{
		i = 8;
		val >>= 8;
	}

	if (val >= 0x10)
	{
		i += 4;
		val >>= 4;
	}
	return OrdBitsTab[val] + i;
}

u8 Bits32(u32 val)
{
	int i = 0;
	if (val >= 0x10000)
	{
		i = 16;
		val >>= 16;
	}

	if (val >= 0x100)
	{
		i += 8;
		val >>= 8;
	}

	if (val >= 0x10)
	{
		i += 4;
		val >>= 4;
	}
	return OrdBitsTab[val] + i;
}

u8 Bits64(u64 val)
{
	if (val > 0xffffffff) return ::Bits32((u32)(val >> 32)) + 32;
	return ::Bits32((u32)val);
}

#endif // USE8BITTAB

///////////////////////////////////////////////////////////////////////////////
// get number of high bits of value (returns top offset of lowest bit + 1: 0..64)

#ifdef USE8BITTAB

u8 HighBits8(u8 val)
{
	return HighOrdBitsTab[val];
}

u8 HighBits16(u16 val)
{
	int i = 16-8;
	if ((val & 0xff) == 0)
	{
		i = 0;
		val >>= 8;
	}
	return HighOrdBitsTab[val & 0xff] + i;
}

u8 HighBits32(u32 val)
{
	int i = 32-8;
	if ((val & 0xffff) == 0)
	{
		i = 16-8;
		val >>= 16;
	}

	if ((val & 0xff) == 0)
	{
		i -= 8;
		val >>= 8;
	}
	return HighOrdBitsTab[val & 0xff] + i;
}

u8 HighBits64(u64 val)
{
	if ((val & 0xffffffff) == 0) return ::HighBits32((u32)(val >> 32));
	return ::HighBits32((u32)val) + 32;
}

#else // USE8BITTAB

u8 HighBits8(u8 val)
{
	int i = 4;
	if ((val & 0x0f) == 0)
	{
		i = 0;
		val >>= 4;
	}
	return HighOrdBitsTab[val & 0x0f] + i;
}

u8 HighBits16(u16 val)
{
	int i = 16-4;
	if ((val & 0xff) == 0)
	{
		i = 8-4;
		val >>= 8;
	}

	if ((val & 0x0f) == 0)
	{
		i -= 4;
		val >>= 4;
	}
	return HighOrdBitsTab[val & 0x0f] + i;
}

u8 HighBits32(u32 val)
{
	int i = 32-4;
	if ((val & 0xffff) == 0)
	{
		i = 16-4;
		val >>= 16;
	}

	if ((val & 0xff) == 0)
	{
		i -= 8;
		val >>= 8;
	}

	if ((val & 0x0f) == 0)
	{
		i -= 4;
		val >>= 4;
	}
	return HighOrdBitsTab[val & 0x0f] + i;
}

u8 HighBits64(u64 val)
{
	if ((val & 0xffffffff) == 0) return ::HighBits32((u32)(val >> 32));
	return ::HighBits32((u32)val) + 32;
}

#endif // USE8BITTAB



// Faster HighBits32

// https://en.wikipedia.org/wiki/De_Bruijn_sequence#Uses

/*

const u8 HighOrdBitsTab2[32] = {
  0, 1, 28, 2, 29, 14, 24, 3, 30, 22, 20, 15, 25, 17, 4, 8, 
  31, 27, 13, 23, 21, 19, 16, 7, 26, 12, 18, 6, 11, 5, 10, 9
};

u8 HighBits32(u32 val)
{
	return 32 - HighOrdBitsTab2[((u32)(val & -(s32)val) * 0x077CB531U) >> 27];
}

*/
